using System;
using System.Data;
using gov.va.med.vbecs.DAL;
using gov.va.med.vbecs;
using TABLE = gov.va.med.vbecs.Common.VbecsTables;
using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;
using Microsoft.Reporting.WinForms;
using System.Reflection;
using System.IO;

namespace gov.va.med.vbecs.BOL
{
    /// <summary>
    /// Summary description for PatientOrder.
    /// </summary>
    public class PatientOrder : BaseBusinessObject
    {

        //
        private  DataTable _patientOrderDataTable;
        //
        private  DataRow  _patientOrderDataRow;
        //
        private DataSet _dsReport;  //CR3574 - removing static to prevent order alert printout mixups
     
        //
        private  Patient _patient;
        //		
		#region constructors


        /// <summary>
        /// Empty constructor - CR3574 - needed a empty constructor in order to call new non-static methods
        /// </summary>
        public PatientOrder() {}    

		///<Developers>
		///	<Developer>COE User</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/20/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3291"> 
		///		<ExpectedInput>DataRow</ExpectedInput>
		///		<ExpectedOutput>PatientOrder object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3292"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// PatientOrder
		/// </summary>
		/// <param name="dr"></param>
		public PatientOrder(DataRow dr)
        {
            this.LoadFromDataRow(dr);
        }

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/20/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3293"> 
		///		<ExpectedInput>PatientOrder Guid, bool</ExpectedInput>
		///		<ExpectedOutput>PatientOrder object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3294"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///	<summary>
		/// Constructor
		/// </summary>
        public PatientOrder(System.Guid patientOrderGuid,
           bool readOnly) 
        {
            this.IsReadOnly = readOnly;
            GetPatientOrderData(patientOrderGuid);
        }

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/20/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3306"> 
		///		<ExpectedInput>PatientOrder Guid</ExpectedInput>
		///		<ExpectedOutput>PatientOrder object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3307"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Constructor
		/// </summary>
		public PatientOrder(System.Guid patientOrderGuid) : this(patientOrderGuid, 
		    true)
		{
		}

		#endregion      

		#region properties

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/28/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6054"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6055"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Patient admitting diagnosis
		/// </summary>
		public string PatientAdmittingDiagnosis
		{
			get
			{
				if (_patientOrderDataRow.IsNull(TABLE.PatientTreatment.PatientAdmittingDiagnosis))
				{
					return string.Empty;
				}

				return (string)_patientOrderDataRow[TABLE.PatientTreatment.PatientAdmittingDiagnosis];
			}
			set
			{
				//
				if (value == string.Empty)
				{
					throw new ArgumentException("Invalid PatientOrderGuid argument", "value");
				}
				_patientOrderDataRow[TABLE.PatientTreatment.PatientAdmittingDiagnosis] = value;
			} 
		}

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/28/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6052"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6053"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Treating specialty name
		/// </summary>
		public string TreatingSpecialtyName
		{
			get
			{
				if (_patientOrderDataRow.IsNull(TABLE.TreatingSpecialty.TreatingSpecialtyName))
				{
					return string.Empty;
				}
				return (string)_patientOrderDataRow[TABLE.TreatingSpecialty.TreatingSpecialtyName];
			}
			set
			{
				//
				if (value == string.Empty)
				{
					throw new ArgumentException("Invalid Treating specialty name argument", "value");
				}
				_patientOrderDataRow[TABLE.TreatingSpecialty.TreatingSpecialtyName] = value;
			} 
		}

        ///<Developers>
        ///	<Developer>Krzysztof Dobranowski</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>4/25/2003</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="2699"> 
        ///		<ExpectedInput>Valid Guid</ExpectedInput>
        ///		<ExpectedOutput>Valid Guid</ExpectedOutput>
        ///	</Case>
		///	
		///<Case type="1" testid ="4093"> 
		///		<ExpectedInput>Empty Guid</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
		/// <summary>
		/// PatientOrderGuid
		/// </summary>
        public System.Guid PatientOrderGuid
        {
            get
            {
                if (_patientOrderDataRow.IsNull(TABLE.PatientOrder.PatientOrderGuid))
                {
                    return System.Guid.Empty;
                }

                return (System.Guid)_patientOrderDataRow[TABLE.PatientOrder.PatientOrderGuid];
            }
            set
            {
                //
                if (value == Guid.Empty)
                {
                    throw new ArgumentException("Invalid PatientOrderGuid argument", "value");
                }
                _patientOrderDataRow[TABLE.PatientOrder.PatientOrderGuid] = value;
            } 

        }
        
		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/20/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3308"> 
		///		<ExpectedInput>A valid Guid</ExpectedInput>
		///		<ExpectedOutput>A valid Guid</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3309"> 
		///		<ExpectedInput>An invalid Guid</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// PatientGuid
		/// </summary>
        public System.Guid PatientGuid
        {
            get
            {
                if (_patientOrderDataRow.IsNull(TABLE.PatientTreatment.PatientGuid))
                {
                    return System.Guid.Empty;
                }

                return (System.Guid)_patientOrderDataRow[TABLE.PatientTreatment.PatientGuid];
            }
            set
            {
                //
                if (value == Guid.Empty)
                {
                    throw new ArgumentException("Invalid PatientGuid argument", "value");
                }
                _patientOrderDataRow[TABLE.PatientTreatment.PatientGuid] = value;
            } 
        }
    
		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/12/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7557"> 
		///		<ExpectedInput>Valid Patient Treatment Guid on non-read only object</ExpectedInput>
		///		<ExpectedOutput>Updated object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7558"> 
		///		<ExpectedInput>Valid Patient Treatment Guid on read only object</ExpectedInput>
		///		<ExpectedOutput>Business object exception thrown</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// PatientTreatmentGuid
		/// </summary>
		public System.Guid PatientTreatmentGuid
		{
			get
			{
				if (_patientOrderDataRow.IsNull(TABLE.PatientOrder.PatientTreatmentGuid))
				{
					return System.Guid.Empty;
				}

				return (System.Guid)_patientOrderDataRow[TABLE.PatientOrder.PatientTreatmentGuid];
			}
			set
			{
				//
				if (value == Guid.Empty)
				{
					throw new ArgumentException("Invalid PatientTreatmentGuid argument", "value");
				}
				_patientOrderDataRow[TABLE.PatientOrder.PatientTreatmentGuid] = value;
			} 
		}

        ///<Developers>
        ///	<Developer>Krzysztof Dobranowski</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>4/25/2003</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="2701"> 
        ///		<ExpectedInput>Valid Patient object</ExpectedInput>
        ///		<ExpectedOutput>Valid Patient object</ExpectedOutput>
        ///	</Case>
		///	
		///<Case type="1" testid ="458"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
		/// <summary>
		/// Patient
		/// </summary>
        public Patient Patient
        {
            get
            {
                if (this._patient == null || this.PatientGuid != _patient.PatientGuid) 
                {
                    this._patient = new Patient(this.PatientGuid);
                }
                return this._patient;
            }
            set
            {
                this._patient = value;
                this.PatientGuid = this.Patient.PatientGuid;
            }
        }


        ///<Developers>
        ///	<Developer>Krzysztof Dobranowski</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>4/25/2003</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="2703"> 
        ///		<ExpectedInput>Valid DateTime</ExpectedInput>
        ///		<ExpectedOutput>Valid DateTime</ExpectedOutput>
        ///	</Case>
        ///
		///<Case type="1" testid ="2704"> 
		///		<ExpectedInput>Invalid DateTime</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
        ///
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
		/// <summary>
		/// OrderPlacedDatetime
		/// </summary>
        public System.DateTime OrderPlacedDatetime
        {
            get
            {
                if (_patientOrderDataRow.IsNull(TABLE.PatientOrder.OrderPlacedDatetime))
                {
                    return System.DateTime.MinValue;
                }

                return (System.DateTime)_patientOrderDataRow[TABLE.PatientOrder.OrderPlacedDatetime];
            }
            set
            {
                //
                if (value == DateTime.MinValue || value == DateTime.MaxValue)
                {
                    throw new ArgumentException("Invalid OrderPlacedDatetime argument", "value");
                }
                _patientOrderDataRow[TABLE.PatientOrder.OrderPlacedDatetime] = value;
            } 

        }
        //

        ///<Developers>
        ///	<Developer>Krzysztof Dobranowski</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>4/25/2003</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="2705"> 
        ///		<ExpectedInput>Valid DateTime</ExpectedInput>
        ///		<ExpectedOutput>Valid DateTime</ExpectedOutput>
        ///	</Case>
        ///
		///<Case type="1" testid ="2706"> 
		///		<ExpectedInput>Invalid DateTime</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
        ///
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
		/// <summary>
		/// OrderReceivedDateTime
		/// </summary>
        public System.DateTime OrderReceivedDateTime
        {
            get
            {
                if (_patientOrderDataRow.IsNull(TABLE.PatientOrder.OrderReceivedDatetime))
                {
                    return System.DateTime.MinValue;
                }

                return (System.DateTime)_patientOrderDataRow[TABLE.PatientOrder.OrderReceivedDatetime];
            }
            set
            {
                //
                if (value == DateTime.MinValue || value == DateTime.MaxValue)
                {
                    throw new ArgumentException("Invalid OrderReceivedDateTime argument", "value");
                }
                _patientOrderDataRow[TABLE.PatientOrder.OrderReceivedDatetime] = value;
            } 

        }


        ///<Developers>
        ///	<Developer>Carl Jensen</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>4/29/2003</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="2718"> 
        ///		<ExpectedInput>Valid OrderGroupNumber</ExpectedInput>
        ///		<ExpectedOutput>Valid OrderGroupNumber</ExpectedOutput>
        ///	</Case>
        ///
        ///<Case type="1" testid ="2719"> 
        ///		<ExpectedInput>Invalid OrderGroupNumber</ExpectedInput>
        ///		<ExpectedOutput>ArgumentException</ExpectedOutput>
        ///	</Case>
        ///
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
		/// <summary>
		/// OrderGroupNumber
		/// </summary>
        public string OrderGroupNumber
        {
            get
            {
                if (_patientOrderDataRow.IsNull(TABLE.PatientOrder.OrderGroupNumber))
                {
                    return "";
                }

                return (string)_patientOrderDataRow[TABLE.PatientOrder.OrderGroupNumber];
            }
            set
            {
                //
                if (value == null || value.Length > 20)
                {
                    throw new ArgumentException("Invalid OrderGroupNumber argument", "value");
                }
                _patientOrderDataRow[TABLE.PatientOrder.OrderGroupNumber] = value;
            } 

        }
        
		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/20/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3315"> 
		///		<ExpectedInput>decimal</ExpectedInput>
		///		<ExpectedOutput>decimal</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="3316"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///<summary>
        /// OrderingProviderId
        ///</summary>
        public string OrderingProviderId
        {
            get
            {
                if (_patientOrderDataRow.IsNull(TABLE.PatientOrder.OrderingProviderId))
                {
                    return string.Empty;
                }

                return _patientOrderDataRow[TABLE.PatientOrder.OrderingProviderId].ToString();
            }
            set
            {
                //
                if (value == string.Empty)
                {
                    throw new ArgumentException("Invalid OrderingProviderId argument", "value");
                }
                _patientOrderDataRow[TABLE.PatientOrder.OrderingProviderId] = value;
            } 

        }

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/20/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3317"> 
		///		<ExpectedInput>A valid string</ExpectedInput>
		///		<ExpectedOutput>A valid string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3318"> 
		///		<ExpectedInput>A invalid string</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// OrderingProviderFirstName
		/// </summary>
        public string OrderingProviderFirstName
        {
            get
            {
                if (_patientOrderDataRow.IsNull(TABLE.PatientOrder.OrderingProviderFirstName))
                {
                    return "";
                }

                return (string)_patientOrderDataRow[TABLE.PatientOrder.OrderingProviderFirstName];
            }
            set
            {
                //
                if (value == null || value.Length > 50)
                {
                    throw new ArgumentException("Invalid OrderingProviderFirstName argument", "value");
                }
                _patientOrderDataRow[TABLE.PatientOrder.OrderingProviderFirstName] = value;
            } 

        }

		///<Developers>
		///	<Developer>Csrl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/20/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3320"> 
		///		<ExpectedInput>A valid middle initial</ExpectedInput>
		///		<ExpectedOutput>A valid middle initial</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3321"> 
		///		<ExpectedInput>An invalid middle initial</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
        /// <summary>
        /// OrderingProviderMiddleInitial
        /// </summary>
        public string OrderingProviderMiddleInitial
        {
            get
            {
                if (_patientOrderDataRow.IsNull(TABLE.PatientOrder.OrderingProviderMiddleInitial))
                {
                    return "";
                }

                return (string)_patientOrderDataRow[TABLE.PatientOrder.OrderingProviderMiddleInitial];
            }
            set
            {
                //
                if (value == null || value.Length > 1)
                {
                    throw new ArgumentException("Invalid OrderingProviderMiddleInitial argument", "value");
                }
                _patientOrderDataRow[TABLE.PatientOrder.OrderingProviderMiddleInitial] = value;
            } 

        }

		///<Developers>
		///	<Developer>Csrl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/20/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3322"> 
		///		<ExpectedInput>A valid last name</ExpectedInput>
		///		<ExpectedOutput>A valid last name</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3323"> 
		///		<ExpectedInput>Null</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
        /// <summary>
        /// OrderingProviderLastName
        /// </summary>
        public string OrderingProviderLastName
        {
            get
            {
                if (_patientOrderDataRow.IsNull(TABLE.PatientOrder.OrderingProviderLastName))
                {
                    return "";
                }

                return (string)_patientOrderDataRow[TABLE.PatientOrder.OrderingProviderLastName];
            }
            set
            {
                //
                if (value == null || value.Length > 30)
                {
                    throw new ArgumentException("Invalid OrderingProviderLastName argument", "value");
                }
                _patientOrderDataRow[TABLE.PatientOrder.OrderingProviderLastName] = value;
            } 

        }

		///<Developers>
		///	<Developer>Csrl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/20/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3324"> 
		///		<ExpectedInput>A valid first name</ExpectedInput>
		///		<ExpectedOutput>A valid first name</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3325"> 
		///		<ExpectedInput>An invalid first name</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
        /// <summary>
        /// OrderEnteredByFirstName
        /// </summary>
        public string OrderEnteredByFirstName
        {
            get
            {
                if (_patientOrderDataRow.IsNull(TABLE.PatientOrder.OrderEnteredByFirstName))
                {
                    return "";
                }
                return (string)_patientOrderDataRow[TABLE.PatientOrder.OrderEnteredByFirstName];
            }
            set
            {
                //
                if (value == null || value.Length > 20)
                {
                    throw new ArgumentException("Invalid OrderEnteredByFirstName argument", "value");
                }
                _patientOrderDataRow[TABLE.PatientOrder.OrderEnteredByFirstName] = value;
            } 

        }

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/20/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3327"> 
		///		<ExpectedInput>A valid middle initial</ExpectedInput>
		///		<ExpectedOutput>A valid middle initial</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3328"> 
		///		<ExpectedInput>An invalid middle initial</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
        /// <summary>
        /// OrderEnteredByMiddleInitial
        /// </summary>
        public string OrderEnteredByMiddleInitial
        {
            get
            {
                if (_patientOrderDataRow.IsNull(TABLE.PatientOrder.OrderEnteredByMiddleInitial))
                {
                    return "";
                }

                return (string)_patientOrderDataRow[TABLE.PatientOrder.OrderEnteredByMiddleInitial];
            }
            set
            {
                //
                if (value == null || value.Length > 1)
                {
                    throw new ArgumentException("Invalid OrderEnteredByMiddleInitial argument", "value");
                }
                _patientOrderDataRow[TABLE.PatientOrder.OrderEnteredByMiddleInitial] = value;
            } 

        }

		///<Developers>
		///	<Developer>Csrl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/20/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3329"> 
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3330"> 
		///		<ExpectedInput>Null</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
        /// <summary>
        /// OrderEnteredByLastName
        /// </summary>
        public string OrderEnteredByLastName
        {
            get
            {
                if (_patientOrderDataRow.IsNull(TABLE.PatientOrder.OrderEnteredByLastName))
                {
                    return "";
                }

                return (string)_patientOrderDataRow[TABLE.PatientOrder.OrderEnteredByLastName];
            }
            set
            {
                //
                if (value == null || value.Length > 30)
                {
                    throw new ArgumentException("Invalid OrderEnteredByLastName argument", "value");
                }
                _patientOrderDataRow[TABLE.PatientOrder.OrderEnteredByLastName] = value;
            } 

        }

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/25/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3365"> 
		///		<ExpectedInput>non zero decimal</ExpectedInput>
		///		<ExpectedOutput>decimal</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3366"> 
		///		<ExpectedInput>0</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
        /// <summary>
        /// OrderEnteredById
        /// </summary>
        public string OrderEnteredById
        {
            get
            {
                if (_patientOrderDataRow.IsNull(TABLE.PatientOrder.OrderEnteredById))
                {
                    return string.Empty;
                }

                return _patientOrderDataRow[TABLE.PatientOrder.OrderEnteredById].ToString();
            }
            set
            {
                //
                if (value == string.Empty)
                {
                    throw new ArgumentException("Invalid OrderEnteredById argument", "value");
                }
                _patientOrderDataRow[TABLE.PatientOrder.OrderEnteredById] = value;
            } 

        }

        ///<Developers>
        ///	<Developer>Krzysztof Dobranowski</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>4/25/2003</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="3367"> 
        ///		<ExpectedInput>A valid string</ExpectedInput>
        ///		<ExpectedOutput>A valid string</ExpectedOutput>
        ///	</Case>
        ///	
		///<Case type="1" testid ="3368"> 
		///		<ExpectedInput>An invalid string</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
		/// <summary>
		/// OrderingHospitalLocation
		/// </summary>
        public string OrderingHospitalLocation
        {
            get
            {
                if (_patientOrderDataRow.IsNull(TABLE.PatientOrder.OrderingHospitalLocation))
                {
                    return "";
                }

                return (string)_patientOrderDataRow[TABLE.PatientOrder.OrderingHospitalLocation];
            }
            set
            {
                //
                if (value == null || value.Length > 20)
                {
                    throw new ArgumentException("Invalid OrderingHospitalLocation argument", "value");
                }
                _patientOrderDataRow[TABLE.PatientOrder.OrderingHospitalLocation] = value;
            } 

        }
        //
        ///<Developers>
        ///	<Developer>Krzysztof Dobranowski</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>4/25/2003</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="2711"> 
        ///		<ExpectedInput>NA</ExpectedInput>
        ///		<ExpectedOutput>A valid string</ExpectedOutput>
        ///	</Case>
		///	
		///<Case type="1" testid ="450"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
		/// <summary>
		/// EnteredBy
		/// </summary>
        public string EnteredBy
        {
            get
            {
                string mi = ' ' + this.OrderEnteredByMiddleInitial;
                return this.OrderEnteredByFirstName + mi.TrimEnd(' ') + ' ' + this.OrderEnteredByLastName;
            }
        }


        ///<Developers>
        ///	<Developer>Krzysztof Dobranowski</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>4/25/2003</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="3369"> 
        ///		<ExpectedInput>A valid string</ExpectedInput>
        ///		<ExpectedOutput>A valid string</ExpectedOutput>
        ///	</Case>
        ///
		///<Case type="1" testid ="3370"> 
		///		<ExpectedInput>An invalid string</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
		/// <summary>
		/// RequestReason
		/// </summary>
        public string RequestReason
        {
            get
            {
                if (_patientOrderDataRow.IsNull(TABLE.PatientOrder.RequestReason))
                {
                    return "";
                }

                return (string)_patientOrderDataRow[TABLE.PatientOrder.RequestReason];
            }
            set
            {
                //
                if (value == null || value.Length > 250)
                {
                    throw new ArgumentException("Invalid RequestReason argument", "value");
                }
                _patientOrderDataRow[TABLE.PatientOrder.RequestReason] = value;
            } 

        }

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/26/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3371"> 
		///		<ExpectedInput>A valid string/ comment</ExpectedInput>
		///		<ExpectedOutput>>A valid string/ comment</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3372"> 
		///		<ExpectedInput>An invalid string/ comment</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
        /// <summary>
        /// OrderComments
        /// </summary>
        public string OrderComments
        {
            get
            {
                if (_patientOrderDataRow.IsNull(TABLE.PatientOrder.OrderComments))
                {
                    return "";
                }

                return (string)_patientOrderDataRow[TABLE.PatientOrder.OrderComments];
            }
            set
            {
                //
                if (value == null || value.Length > 255)
                {
                    throw new ArgumentException("Invalid OrderComments argument", "value");
                }
                _patientOrderDataRow[TABLE.PatientOrder.OrderComments] = value;
            } 

        }

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/26/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3373"> 
		///		<ExpectedInput>bool</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="454"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// InformedConsentIndicator
		/// </summary>
        public bool InformedConsentIndicator
        {
            get
            {
                if (_patientOrderDataRow.IsNull(TABLE.PatientOrder.InformedConsentIndicator))
                {
                    return false;
                }

                return (bool)_patientOrderDataRow[TABLE.PatientOrder.InformedConsentIndicator];
            }
            set
            {
                _patientOrderDataRow[TABLE.PatientOrder.InformedConsentIndicator] = value;
            } 

        }
      
		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/26/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3375"> 
		///		<ExpectedInput>A valid string/ surgery name</ExpectedInput>
		///		<ExpectedOutput>A valid string/ surgery name</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3376"> 
		///		<ExpectedInput>An invalid string</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// SurgeryName
		/// </summary>
        public string SurgeryName
        {
            get
            {
                if (_patientOrderDataRow.IsNull(TABLE.PatientOrder.SurgeryName))
                {
                    return "";
                }

                return (string)_patientOrderDataRow[TABLE.PatientOrder.SurgeryName];
            }
            set
            {
                //
                if (value == null || value.Length > 50)
                {
                    throw new ArgumentException("Invalid SurgeryName argument", "value");
                }
                _patientOrderDataRow[TABLE.PatientOrder.SurgeryName] = value;
            } 

        }
        //
		
		

		#endregion
        //
        #region 
	
		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/20/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6458"> 
		///		<ExpectedInput>Valid string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="6459"> 
		///		<ExpectedInput>Valid string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6460"> 
		///		<ExpectedInput>InValid string, longer than 20 characters.</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6461"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// DivisionCode
		/// </summary>
        public string DivisionCode
        {
            get
            {
                if (_patientOrderDataRow.IsNull(TABLE.PatientOrder.DivisionCode))
                {
                    return "";
                }

                return (string)_patientOrderDataRow[TABLE.PatientOrder.DivisionCode];
            }
            set
            {
                //
                if (value == null || value.Length > 10)
                {
                    throw new ArgumentException("Invalid DivisionCode argument", "value");
                }
                _patientOrderDataRow[TABLE.PatientOrder.DivisionCode] = value;
            } 

        }
        //
	
		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/20/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6466"> 
		///		<ExpectedInput>Valid date</ExpectedInput>
		///		<ExpectedOutput>Date</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="6467"> 
		///		<ExpectedInput>Valid Date</ExpectedInput>
		///		<ExpectedOutput>Date</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6470"> 
		///		<ExpectedInput>Invalid Date</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// LastUpdateDate
		/// </summary>
        public new System.DateTime LastUpdateDate
        {
            get
            {
                if (_patientOrderDataRow.IsNull(TABLE.PatientOrder.LastUpdateDate))
                {
                    return System.DateTime.MinValue;
                }

                return (System.DateTime)_patientOrderDataRow[TABLE.PatientOrder.LastUpdateDate];
            }
            set
            {
                //
                if (value == DateTime.MinValue || value == DateTime.MaxValue)
                {
                    throw new ArgumentException("Invalid LastUpdateDate argument", "value");
                }
                _patientOrderDataRow[TABLE.PatientOrder.LastUpdateDate] = value;
            } 

        }

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/20/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6471"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6472"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="6473"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="6474"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
        /// <summary>
        /// LastUpdateUser
        /// </summary>	
        public new string LastUpdateUser
        {
            get
            {
                if (_patientOrderDataRow.IsNull(TABLE.PatientOrder.LastUpdateUser))
                {
                    return "";
                }

                return (string)_patientOrderDataRow[TABLE.PatientOrder.LastUpdateUser];
            }
            set
            {
                //
                if (value == null || value.Length > 30)
                {
                    throw new ArgumentException("Invalid LastUpdateUser argument", "value");
                }
                _patientOrderDataRow[TABLE.PatientOrder.LastUpdateUser] = value;
            } 

        }
	
		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/20/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6475"> 
		///		<ExpectedInput>byte[]</ExpectedInput>
		///		<ExpectedOutput>byte[]</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="6476"> 
		///		<ExpectedInput>byte[]</ExpectedInput>
		///		<ExpectedOutput>byte[]</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6477"> 
		///		<ExpectedInput>InValid: null</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6478"> 
		///		<ExpectedInput>InValid: byte[] longer than 8</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// RowVersion
		/// </summary>
        public new System.Byte[] RowVersion
        {
            get
            {
                if (_patientOrderDataRow.IsNull(TABLE.PatientOrder.RowVersion))
                {
                    return null;
                }

                return (System.Byte[])_patientOrderDataRow[TABLE.PatientOrder.RowVersion];
            }
            set
            {
                //
                if (value == null || value.Length > 8)
                {
                    throw new ArgumentException("Invalid RowVersion argument", "value");
                }
                _patientOrderDataRow[TABLE.PatientOrder.RowVersion] = value;
            } 

        }
        //

      
        #endregion
        //
        #region Implementation of IBaseBusinessObject

		///<Developers>
		///	<Developer>COE User</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/25/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3363"> 
		///		<ExpectedInput>DataRow</ExpectedInput>
		///		<ExpectedOutput>DataRow</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3364"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		///  Loads a data row from an object
		/// </summary>
        public override System.Data.DataRow LoadDataRowFromThis(System.Data.DataRow dataRow)
        {
            return null;
        }
        /// <summary>
        /// Loads an object from a data row 
        /// </summary>
        protected sealed override void LoadFromDataRow(System.Data.DataRow dataRow)
        {
            _patientOrderDataTable = dataRow.Table;
            _patientOrderDataRow = dataRow;

            // overriddem properties
            if (dataRow.Table.Columns.Contains(TABLE.PatientOrder.PatientOrderGuid))
            {
                this.PatientOrderGuid = dataRow.IsNull(TABLE.PatientOrder.PatientOrderGuid) ? Guid.NewGuid() : (System.Guid) dataRow[TABLE.PatientOrder.PatientOrderGuid];
            }
			if (dataRow.Table.Columns.Contains(TABLE.PatientOrder.PatientTreatmentGuid))
			{
				this.PatientTreatmentGuid = dataRow.IsNull(TABLE.PatientOrder.PatientTreatmentGuid) ? Guid.NewGuid() : (System.Guid) dataRow[TABLE.PatientOrder.PatientTreatmentGuid];
			}
			if (dataRow.Table.Columns.Contains(TABLE.PatientOrder.OrderReceivedDatetime))
            {
                this.OrderReceivedDateTime = dataRow.IsNull(TABLE.PatientOrder.OrderReceivedDatetime) ? DateTime.MinValue : (DateTime) dataRow[TABLE.PatientOrder.OrderReceivedDatetime];
            }
            if (dataRow.Table.Columns.Contains(TABLE.PatientOrder.OrderingProviderLastName))
            {
                this.OrderingProviderLastName = dataRow.IsNull(TABLE.PatientOrder.OrderingProviderLastName) ? String.Empty : dataRow[TABLE.PatientOrder.OrderingProviderLastName].ToString();
            }
            if (dataRow.Table.Columns.Contains(TABLE.PatientOrder.OrderingProviderFirstName))
            {
                this.OrderingProviderFirstName = dataRow.IsNull(TABLE.PatientOrder.OrderingProviderFirstName) ? String.Empty : dataRow[TABLE.PatientOrder.OrderingProviderFirstName].ToString();
            }
			if (dataRow.Table.Columns.Contains(TABLE.PatientOrder.OrderingProviderMiddleInitial))
			{
				this.OrderingProviderMiddleInitial = dataRow.IsNull(TABLE.PatientOrder.OrderingProviderMiddleInitial) ? String.Empty : dataRow[TABLE.PatientOrder.OrderingProviderMiddleInitial].ToString();
			}
			if (dataRow.Table.Columns.Contains(TABLE.PatientOrder.DivisionCode))
			{
				this.DivisionCode = dataRow.IsNull(TABLE.PatientOrder.DivisionCode) ? String.Empty : dataRow[TABLE.PatientOrder.DivisionCode].ToString();
			}
        }
        //
		#endregion
        //
        #region methods


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/6/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5266"> 
		///		<ExpectedInput>StartDate, EndDate, SpecimenUid, SSN, PatientName</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5267"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NoRecordsFoundException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// GetOrderSearchDetails
		/// </summary>
		/// <param name="startDate"></param>
		/// <param name="endDate"></param>
		/// <param name="specimenUid"></param>
		/// <param name="patientSsn"></param>
		/// <param name="patientLastName"></param>
		/// <returns></returns>
		public static DataTable GetOrderSearchDetails(DateTime startDate, DateTime endDate, string specimenUid, string patientSsn, string patientLastName)
		{
			DataTable dtDetails = DAL.PatientOrder.GetOrderSearchDetails(startDate, endDate, specimenUid, patientSsn, patientLastName);
			
			if (dtDetails.Rows.Count == 0)
			{
				throw new BOL.NoRecordsFoundException();
			}
			
			return dtDetails;
		}

        ///<Developers>
        ///	<Developer>Krzysztof Dobranowski</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>4/25/2003</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="2715"> 
        ///		<ExpectedInput>NA</ExpectedInput>
        ///		<ExpectedOutput>The DataTable containing the list of order priorities</ExpectedOutput>
        ///	</Case>
		///	
		///<Case type="1" testid ="451"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
        /// <summary>
        /// Get the list of order priorities
        /// </summary>
        /// <returns></returns>
        public static DataTable GetOrderUrgencyList()
        {
            return DAL.PatientOrder.GetOrderUrgencyList();
        }

		///<Developers>
		///	<Developer>Margaret Jablonski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/10/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3001"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>An empty PatientOrder datatable</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="455"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Initializes an empty PatientOrder data table
		/// </summary>
		/// <returns>datatable</returns>
		public static DataTable InitializePatientOrderTable()
		{
			DataTable dtPatientOrder = new DataTable(TABLE.PatientOrder.TableName);
			dtPatientOrder.Columns.Add(TABLE.PatientOrder.PatientOrderGuid, typeof(Guid));
			dtPatientOrder.Columns.Add(TABLE.PatientTreatment.PatientGuid, typeof(Guid));
			dtPatientOrder.Columns.Add(TABLE.PatientOrder.PatientTreatmentGuid, typeof(Guid));
			dtPatientOrder.Columns.Add(TABLE.PatientOrder.OrderGroupNumber, typeof(string));
			dtPatientOrder.Columns.Add(TABLE.PatientOrder.OrderPlacedDatetime, typeof(DateTime));
			dtPatientOrder.Columns.Add(TABLE.PatientOrder.OrderReceivedDatetime, typeof(DateTime));
			//Removed - seems to have vanished from database!
			//dtPatientOrder.Columns.Add(TABLE.PatientOrder.LabOrderNumber, typeof(string));			
			dtPatientOrder.Columns.Add(TABLE.PatientOrder.OrderingProviderId);			
			dtPatientOrder.Columns.Add(TABLE.PatientOrder.OrderingProviderLastName, typeof(string));			
			dtPatientOrder.Columns.Add(TABLE.PatientOrder.OrderingProviderFirstName, typeof(string));			
			dtPatientOrder.Columns.Add(TABLE.PatientOrder.OrderingProviderMiddleInitial, typeof(string));			
			dtPatientOrder.Columns.Add(TABLE.PatientOrder.OrderingHospitalLocation, typeof(string));
			dtPatientOrder.Columns.Add(TABLE.PatientOrder.OrderEnteredById);
			dtPatientOrder.Columns.Add(TABLE.PatientOrder.OrderEnteredByLastName, typeof(string));
			dtPatientOrder.Columns.Add(TABLE.PatientOrder.OrderEnteredByFirstName, typeof(string));
			dtPatientOrder.Columns.Add(TABLE.PatientOrder.OrderEnteredByMiddleInitial, typeof(string));
			dtPatientOrder.Columns.Add(TABLE.PatientOrder.RequestReason, typeof(string));
			dtPatientOrder.Columns.Add(TABLE.PatientOrder.SpecimenRequiredIndicator, typeof(byte)); 
			dtPatientOrder.Columns.Add(TABLE.PatientOrder.OrderComments, typeof(string)); 
			dtPatientOrder.Columns.Add(TABLE.PatientOrder.InformedConsentIndicator, typeof(byte));
			dtPatientOrder.Columns.Add(TABLE.PatientOrder.SurgeryName, typeof(string));
			dtPatientOrder.Columns.Add(TABLE.PatientOrder.DivisionCode, typeof(string));
			dtPatientOrder.Columns.Add(TABLE.PatientOrder.LastUpdateUser, typeof(string));

			return dtPatientOrder;
		}

		/// <summary>
		/// Gets patient order data
		/// </summary>
		/// <param name="patientOrderGuid"></param>
		protected void GetPatientOrderData(System.Guid patientOrderGuid)
		{
			_patientOrderDataTable = DAL.PatientOrder.GetPatientOrder(patientOrderGuid);
			if (_patientOrderDataTable.Rows.Count > 0) 
			{
				_patientOrderDataRow = _patientOrderDataTable.Rows[0];
			}
			else 
			{
				//throw new BusinessObjectException("There is no patient order with this GUID");
				MakeNew();
			}
			_patientOrderDataTable.ColumnChanged += new DataColumnChangeEventHandler(this.OnColumnChanged);
		}
		//
		private void MakeNew()
		{
			//
			_patientOrderDataRow = _patientOrderDataTable.NewRow();
			_patientOrderDataRow[TABLE.PatientOrder.PatientOrderGuid] = Guid.NewGuid();
			_patientOrderDataRow[TABLE.PatientTreatment.PatientGuid] = Guid.Empty;
			_patientOrderDataRow[TABLE.PatientOrder.LastUpdateDate] = VBECSDateTime.GetDivisionCurrentDateTime();
			_patientOrderDataRow[TABLE.PatientOrder.LastUpdateUser] = String.Empty;
			_patientOrderDataRow[TABLE.PatientOrder.RowVersion] = new byte[8];
			_patientOrderDataTable.Rows.Add(_patientOrderDataRow);

			//
			this.PatientOrderGuid = Guid.NewGuid();
			this.LastUpdateUser = Common.LogonUser.LogonUserName;
			this.IsNew = true;
			this.IsReadOnly = false;
		}
		// 
		private void OnColumnChanged(object sender, DataColumnChangeEventArgs eventArgs)
		{
			if (this.IsReadOnly && !this.IsNew) 
			{
				throw new BusinessObjectException("Object is marked read-only");
			}
			this.IsDirty = true;
		}



		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/20/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6481"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6482"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Static method will retrieve patient order group information and print to a printer,
		/// according to division alert settings.
		/// </summary>
		/// <param name="patientOrderGuid">Patient order unique identifier</param>
		/// <param name="userName">User name</param>
		/// <param name="divisionCode">Division code</param>
		public static void PrintPatientOrderAlert(Guid patientOrderGuid, string userName, string divisionCode)
		{
            //CR3574 - PatientOrder methods no longer static to avoid order alert Printout corruption
			new BOL.PatientOrder().PrintPatientOrderAlert(patientOrderGuid, string.Empty, userName, divisionCode);
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/20/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6479"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6480"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Static method will retrieve patient order group information and print to a printer,
		/// according to division alert settings.
		/// </summary>
		/// <param name="patientOrderGuid">Patient order unique identifier</param>
		/// <param name="cprsOrderNumber">Cprs Order Number</param>
		/// <param name="userName">User name</param>
		/// <param name="divisionCode">Division code</param>
		public void PrintPatientOrderAlert(Guid patientOrderGuid, string cprsOrderNumber, string userName, string divisionCode)
		{
			string orderComments = "";
			//
			BOL.Division dvsn = new BOL.Division(divisionCode);
			//
			bool isPrinterConfigured = BOL.VbecsReport.IsPrinterConfigured(dvsn.DivisionAlerts.PrinterName);
			//
			if ((dvsn.DivisionAlerts.PrintAlertsIndicator)&&(isPrinterConfigured))
			{
				//We cannot use Common.LogonUser.LogonUserDivisionCode because the HL7 service is not logged in to Vbecs
				DataSet dsHeader =  BOL.VbecsReport.GetReportHeader(userName,divisionCode);
				//
				_dsReport = DAL.PatientOrder.PrintPatientOrderAlert(patientOrderGuid,cprsOrderNumber,divisionCode);
				//
                _dsReport.Tables[0].TableName = TABLE.PatientOrder.TableName;
                _dsReport.Tables[0].Columns.Add(TABLE.ExceptionPatientTest.PatientName, typeof(string));
                _dsReport.Tables[0].Columns.Add(ARTIFICIAL.OrderingProviderName, typeof(string));
                _dsReport.Tables[1].TableName = TABLE.OrderedTest.TableName;
                _dsReport.Tables[2].TableName = TABLE.OrderedComponent.TableName;
                // RTC defect 230461 
                if (!dvsn.DivisionAlerts.IsPatientOrderPrintAlertRequired(_dsReport.Tables[0], _dsReport.Tables[2], _dsReport.Tables[1], divisionCode))
                {
                    return;
                }

				//Retrieve PatientOrder.OrderComments from any of the ordered component or ordered test entries (All comments are the same!)
                foreach (DataRow dr in _dsReport.Tables[2].Rows)
				{
					if (dr[TABLE.OrderedComponent.OrderedComponentComments].ToString().Length>0)
					{
						orderComments = dr[TABLE.OrderedComponent.OrderedComponentComments].ToString();
						break;
					}
				}
				if (orderComments.Length==0)
				{
                    foreach (DataRow dr in _dsReport.Tables[1].Rows)
					{
						if (dr[TABLE.OrderedTest.OrderedTestComments].ToString().Length>0)
						{
							orderComments = dr[TABLE.OrderedTest.OrderedTestComments].ToString();
							break;
						}
					}
				}
				//
                foreach (DataRow dr in _dsReport.Tables[0].Rows)
				{
					dr[TABLE.ExceptionPatientTest.PatientName] = BOL.Patient.BuildDisplayName(dr[TABLE.Patient.PatientFirstName].ToString(),
						dr[TABLE.Patient.PatientMiddleName].ToString(),dr[TABLE.Patient.PatientLastName].ToString());
					dr[ARTIFICIAL.OrderingProviderName] = BOL.Patient.BuildDisplayName(dr[TABLE.PatientOrder.OrderingProviderFirstName].ToString(),
						dr[TABLE.PatientOrder.OrderingProviderMiddleInitial].ToString(),dr[TABLE.PatientOrder.OrderingProviderLastName].ToString());
					dr[TABLE.PatientOrder.OrderComments] = orderComments;
				}
#if DEBUG
				//_dsReport.Tables[Common.VbecsTables.PatientOrder.TableName].WriteXml("e:\\temp\\PatientOrderAlertDataSet.xml", XmlWriteMode.WriteSchema);
                //_dsReport.Tables[Common.VbecsTables.OrderedTest.TableName].WriteXml("e:\\temp\\DiagnosticTestAlertDataSet.xml", XmlWriteMode.WriteSchema);
                //_dsReport.Tables[Common.VbecsTables.OrderedComponent.TableName].WriteXml("e:\\temp\\ComponentOrderAlertDataSet.xml", XmlWriteMode.WriteSchema);
#endif
				//
                ReportDataSource reportSourceMain = new ReportDataSource();
                if (_dsReport.Tables.Count > 0)
                {
                    reportSourceMain.Name = "PatientOrderAlertDataSet";
                    reportSourceMain.Value = _dsReport.Tables[Common.VbecsTables.PatientOrder.TableName];
                }

                ReportDataSource reportSourceHeader = new ReportDataSource();
                if (dsHeader.Tables.Count > 0)
                {
                    reportSourceHeader.Name = "ReportHeaderDataSet";
                    reportSourceHeader.Value = dsHeader.Tables[0];
                }

                //CR 3331
                Assembly assembly = Assembly.LoadFrom(Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location) + "\\Reports.dll");
                Stream stream = assembly.GetManifestResourceStream("gov.va.med.vbecs.reports.DisplayUserAlerts.PatientOrderAlert.rdlc");
                Stream streamOrderedTest = assembly.GetManifestResourceStream("gov.va.med.vbecs.reports.DisplayUserAlerts.DiagnosticTestAlert.rdlc");
                Stream streamOrderedComponent = assembly.GetManifestResourceStream("gov.va.med.vbecs.reports.DisplayUserAlerts.ComponentOrderAlert.rdlc");

                LocalReport report = new LocalReport();
                report.SubreportProcessing += new SubreportProcessingEventHandler(LoadSubreportsEventHandler);

                report.DataSources.Add(reportSourceHeader);
                report.DataSources.Add(reportSourceMain);
                report.LoadReportDefinition(stream);
                
                report.LoadSubreportDefinition("OrderedTestAlert", streamOrderedTest);
                report.LoadSubreportDefinition("OrderedComponentAlert", streamOrderedComponent);

                if (_dsReport.Tables[Common.VbecsTables.OrderedTest.TableName].Rows.Count > 0)
                    report.SetParameters(new ReportParameter("IncludeOrderedTest", "true"));
                else
                    report.SetParameters(new ReportParameter("IncludeOrderedTest", "false"));

                if (_dsReport.Tables[Common.VbecsTables.OrderedComponent.TableName].Rows.Count > 0)
                    report.SetParameters(new ReportParameter("IncludeOrderedComponent", "true"));
                else
                    report.SetParameters(new ReportParameter("IncludeOrderedComponent", "false"));

                BOL.VbecsReportPrinter reportPrinter = new VbecsReportPrinter(dvsn.DivisionAlerts.PrinterName, false, report);
                reportPrinter.Print();
			}
		}

        private void LoadSubreportsEventHandler(object sender, SubreportProcessingEventArgs e)
        {
            if (e.ReportPath == "OrderedTestAlert" || e.DataSourceNames[0].ToString() == "OrderedTestAlertDataSet")
            {
                ReportDataSource reportOrderedTest = new ReportDataSource();
                if (_dsReport.Tables.IndexOf(Common.VbecsTables.OrderedTest.TableName) != -1)
                {
                    reportOrderedTest.Name = "OrderedTestAlertDataSet";
                    reportOrderedTest.Value = _dsReport.Tables[Common.VbecsTables.OrderedTest.TableName];
                }
                e.DataSources.Add(reportOrderedTest);
            }
            else if (e.ReportPath == "OrderedComponentAlert" || e.DataSourceNames[0].ToString() == "OrderedComponentAlertDataSet")
            {
                ReportDataSource reportOrderedComponent = new ReportDataSource();
                if (_dsReport.Tables.IndexOf(Common.VbecsTables.OrderedComponent.TableName) != -1)
                {
                    reportOrderedComponent.Name = "OrderedComponentAlertDataSet";
                    reportOrderedComponent.Value = _dsReport.Tables[Common.VbecsTables.OrderedComponent.TableName];
                }
                e.DataSources.Add(reportOrderedComponent);
            }
        }

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/20/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6483"> 
		///		<ExpectedInput>Guid</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6484"> 
		///		<ExpectedInput>Empty Guid</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// GetReflexTestsForOrder
		/// </summary>
		/// <param name="orderGuid">OrderedComponentGuid or OrderedTestGuid</param>
		/// <returns></returns>
		public static DataTable GetReflexTestsForOrder(Guid orderGuid)
		{	
			if (orderGuid == Guid.Empty)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("orderGuid").ResString);
			}

			return DAL.PatientOrder.GetReflexTestsForOrder(orderGuid);
		}

		#endregion
    }
}
